from typing import Any
from uuid import uuid4
import httpx
from a2a.client import A2ACardResolver, A2AClient
from a2a.types import (
    MessageSendParams,
    SendMessageRequest,
)

# メイン
async def main() -> None:
    # 定数
    BASE_URL = 'http://localhost:9999'
    PUBLIC_AGENT_CARD_PATH = '/.well-known/agent.json'

    async with httpx.AsyncClient() as httpx_client:
        # A2ACardResolverの準備
        resolver = A2ACardResolver(
            httpx_client=httpx_client,
            base_url=BASE_URL,
            # デフォルトパスのためpath指定は省略
        )

        # エージェントカードの取得
        try:
            print(f'Agent Card の取得中: {BASE_URL}{PUBLIC_AGENT_CARD_PATH}')
            _public_card = (
                await resolver.get_agent_card()
            )
            print('エージェントカードの取得')
            print(_public_card.model_dump_json(indent=2, exclude_none=True))
        except Exception as e:
            raise RuntimeError('エージェントカードの取得失敗') from e

        # A2AClientの準備
        client = A2AClient(httpx_client=httpx_client, agent_card=_public_card)

        # メッセージの定義
        send_message_payload: dict[str, Any] = {
            'message': {
                'role': 'user',
                'parts': [
                    {'kind': 'text', 'text': 'how much is 10 USD in INR?'}
                ],
                'messageId': uuid4().hex,
            },
        }

        # メッセージの送信
        request = SendMessageRequest(
            id=str(uuid4()), params=MessageSendParams(**send_message_payload)
        )
        response = await client.send_message(request)
        print(response.model_dump(mode='json', exclude_none=True))

# メイン
if __name__ == '__main__':
    import asyncio
    asyncio.run(main())