﻿using UnityEngine;
using UnityEngine.InputSystem;

[RequireComponent(typeof(PlayerInput))]       // なければ自動で追加
[RequireComponent(typeof(Rigidbody))]         // なければ自動で追加
[RequireComponent(typeof(CapsuleCollider))]   // なければ自動で追加

// キー入力で移動＋ジャンプ（重力あり）
public class OnKeyPressMoveGravity : MonoBehaviour
{
    //-------------------------------------
    // 移動スピード
    public float speed = 5f;
    // ジャンプの強さ
    public float jumppower = 8f;
    // 地面との距離を調べる長さ（これ以下なら着地とみなす）
    public float checkDistance = 0.1f;
    // ジャンプや着地判定が安定しないときの調整値
    public float footOffset = 0.01f;
    //-------------------------------------
    private Rigidbody rbody;
    private CapsuleCollider col;
    private Vector2 moveInput;
    private bool jumpRequested = false;
    private bool isGrounded = false;
    private bool isJumping = false;

    void Awake()
    {
        // Rigidbody を取得し、回転しないよう固定
        rbody = GetComponent<Rigidbody>();
        col = GetComponent<CapsuleCollider>();
        rbody.constraints = RigidbodyConstraints.FreezeRotation;
    }

    public void OnMove(InputValue value)
    {
        // 移動の入力（上下左右キーやWASD）を受け取る
        moveInput = value.Get<Vector2>();
    }

    public void OnJump()
    {
        // 着地していて、まだジャンプ中でなければジャンプを予約
        if (isGrounded && !isJumping)
            jumpRequested = true;
    }

    void Update()
    {
        // 足元から下方向に Ray を飛ばして「着地しているか」を判定
        Vector3 footPos = new Vector3(
            transform.position.x,
            col.bounds.min.y + footOffset,
            transform.position.z
        );
        isGrounded = Physics.Raycast(footPos, Vector3.down, checkDistance);

        // 下向きに動いていたら「ジャンプ終了」と判定
        if (rbody.linearVelocity.y <= 0)
            isJumping = false;
    }

    void FixedUpdate()
    {
        // 入力方向にキャラの向きを合わせる
        Vector3 dir = new Vector3(moveInput.x, 0, moveInput.y);
        if (dir.sqrMagnitude > 0.1f)
        {
            transform.rotation = Quaternion.LookRotation(dir);
        }

        // 移動（物理的に速度を与えて動かす）
        Vector3 vel = rbody.linearVelocity;
        Vector3 move = dir * speed;
        rbody.linearVelocity = new Vector3(move.x, vel.y, move.z);

        // ジャンプ入力があれば、瞬間的に上方向へ力を加える
        if (jumpRequested)
        {
            rbody.AddForce(Vector3.up * jumppower, ForceMode.Impulse);
            jumpRequested = false;
            isJumping = true;
        }
    }
}
