using System.Collections.Generic;
using UnityEngine;

[RequireComponent(typeof(Rigidbody))]   // なければ自動で追加
[RequireComponent(typeof(BoxCollider))] // なければ自動で追加

// ずっと、往復移動する（乗れる床）
public class ForeverPingPongFloor : MonoBehaviour
{
    //-------------------------------------
    // 移動スピード（X=左右、Y=上下、Z=前後）
    public Vector3 speed = new Vector3(1, 0, 0);
    // 片道の移動時間（秒）
    public float moveTime = 2.0f;
    //-------------------------------------
    private float dir = 1.0f;           // 移動方向（1 または -1）
    private Rigidbody rbody;            // この床の Rigidbody
    private float timer = 0.0f;         // 移動時間のカウント
    private List<Rigidbody> riders = new List<Rigidbody>(); // 上に乗っているオブジェクトのリスト

    void Awake()
    {
        // Rigidbody を取得して、物理的な力の影響を受けない設定にする
        rbody = GetComponent<Rigidbody>();
        rbody.isKinematic = true;
    }

    void FixedUpdate()
    {
        // 一定時間ごとに移動方向を反転する
        timer += Time.fixedDeltaTime;
        if (timer >= moveTime)
        {
            dir = -dir;
            timer = 0.0f;
        }

        // 床の次の位置を計算して移動
        Vector3 nextPos = transform.position + speed * dir * Time.fixedDeltaTime;
        Vector3 deltaMove = nextPos - transform.position;
        rbody.MovePosition(nextPos);

        // 上に乗っているオブジェクトも一緒に移動させる
        foreach (var body in riders)
        {
            if (body)
            {
                body.MovePosition(body.position + deltaMove);
            }
        }
    }

    void OnCollisionStay(Collision collision)
    {
        // 床に接触しているオブジェクトをリストに追加
        Rigidbody body = collision.rigidbody;
        if (body && !riders.Contains(body))
        {
            riders.Add(body);
        }
    }

    void OnCollisionExit(Collision collision)
    {
        // 接触が終わったオブジェクトをリストから削除
        Rigidbody body = collision.rigidbody;
        if (body)
        {
            riders.Remove(body);
        }
    }
}
