package jp.study.game;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.SurfaceTexture;
import android.view.MotionEvent;
import android.view.TextureView;
import android.view.View;

import java.util.ArrayList;

/**
 * ゲーム画面を描画するためのTextureView
 */
public class GameView extends TextureView implements TextureView.SurfaceTextureListener, View.OnTouchListener {
    volatile private boolean mIsRunnable;
    volatile private float mTouchedX;
    volatile private float mTouchedY;
    private Thread mThread;

    /**
     * スーパークラスにはデフォルトコンストラクターがないため引数付きのコンストラクターを明示的に呼び出す * @param context Activity
     */
    public GameView(Context context) {
        super(context); // 親クラスのコンストラクターを引数 contextで呼び出す
        setSurfaceTextureListener(this);
        setOnTouchListener(this);
    }


    public void start() {
        mThread = new Thread(new Runnable() {
            @Override
            public void run() {
                Paint paint = new Paint();
                paint.setColor(Color.RED);
                paint.setStyle(Paint.Style.FILL);
                while (true) {
                    // アプリ実行中繰り返し呼ばれる
                    synchronized (GameView.this) {
                        if (!mIsRunnable) {
                            break; // ループを終了する
                        }
                        Canvas canvas = lockCanvas();
                        if (canvas == null) {
                            continue; // キャンバスが取得できない時はループのやり直し
                        }
                        canvas.drawColor(Color.BLACK);
                        for (Block item : mItemList) {
                            // mItemListの内容が1つずつblockに渡される
                            item.draw(canvas, paint);
                        }
                        unlockCanvasAndPost(canvas);
                    }
                }
            }
        });
        mIsRunnable = true;
        mThread.start();
    }

    public void stop() {
        mIsRunnable = false;
    }

    @Override
    public void onSurfaceTextureAvailable(SurfaceTexture surface, int width, int height) {
        readyObjects(width, height);
    }
    @Override
    public void onSurfaceTextureSizeChanged(SurfaceTexture surface, int width, int height) {
        readyObjects(width, height);
    }

    private ArrayList<Block> mItemList;
    public void readyObjects(int width, int height) {
        float blockWidth = width / 10;
        float blockHeight = height / 20;
        mItemList = new ArrayList<Block>();
        for (int i = 0; i < 100; i++) {
            float blockTop = i / 10 * blockHeight;
            float blockLeft = i % 10 * blockWidth;
            float blockBottom = blockTop + blockHeight;
            float blockRight = blockLeft + blockWidth;
            mItemList.add(new Block(blockTop, blockLeft, blockBottom, blockRight));
        }
    }

    @Override
    public boolean onSurfaceTextureDestroyed(SurfaceTexture surface) {
        synchronized (this) {
            return true;
        }
    }

    @Override
    public void onSurfaceTextureUpdated(SurfaceTexture surface) {

    }

    @Override
    public boolean onTouch(View v, MotionEvent event) {
        mTouchedX = event.getX();
        mTouchedY = event.getY();
        return true;
    }


}