﻿
namespace XELF.Framework {

	/// <summary>
	/// クォータニオン（四元数）
	/// </summary>
	public partial struct Quaternion {
		/// <summary>
		/// X
		/// </summary>
		public float X;
		/// <summary>
		/// Y
		/// </summary>
		public float Y;
		/// <summary>
		/// Z
		/// </summary>
		public float Z;
		/// <summary>
		/// W
		/// </summary>
		public float W;
	}

	public partial struct Quaternion {
		static readonly Quaternion identity = new Quaternion(0, 0, 0, 1);
		/// <summary>
		/// 単位元 Quaternion(0, 0, 0, 1)
		/// </summary>
		public static Quaternion Identity { get { return identity; } }
	}

	public partial struct Quaternion {
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="x">X</param>
		/// <param name="y">Y</param>
		/// <param name="z">Z</param>
		/// <param name="w">W</param>
		public Quaternion(float x, float y, float z, float w) {
			X = x;
			Y = y;
			Z = z;
			W = w;
		}
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="vectorPart">ベクトル部分</param>
		/// <param name="scalarPart">スカラー部分</param>
		public Quaternion(Vector3 vectorPart, float scalarPart) {
			X = vectorPart.X;
			Y = vectorPart.Y;
			Z = vectorPart.Z;
			W = scalarPart;
		}

		/// <summary>
		/// 軸周りの回転からクォータニオンを作る。
		/// </summary>
		/// <param name="axis">軸</param>
		/// <param name="angle">角度[Radian]</param>
		/// <returns></returns>
		public static Quaternion CreateFromAxisAngle(Vector3 axis, float angle) {
			axis.Normalize();
			angle *= 0.5f;
			var s = (float)System.Math.Sin(angle);
			var c = (float)System.Math.Cos(angle);
			Vector3.Multiply(ref axis, s, out axis);
			return new Quaternion(axis, c);
		}

		/// <summary>
		/// 文字列表現を得る。
		/// </summary>
		/// <returns></returns>
		public override string ToString() {
			return string.Format(System.Globalization.CultureInfo.InvariantCulture, "{0},{1},{2},{3}", X, Y, Z, W);
		}
	}
}