-- -*- indent-tabs-mode: nil -*-
module Main(main) where 

import Control.Monad
import Data.List
import Graphics.UI.WX
import System.Random

type Pt = (Int, Int)
type Board = [[Color]]
type Score = Int

mapBoard :: (Pt -> a -> b) -> [[a]] -> [[b]]
mapBoard f board = zipWith (\x column -> zipWith (\y v -> f (x, y) v) [0..] column) [0..] board
filterBoard :: (Pt -> a -> Bool) -> [[a]] -> [[a]]
filterBoard f board = filter ((/=0).length) $ zipWith (\x column -> concat $ zipWith (\y v -> if f (x, y) v then [v] else []) [0..] column) [0..] board


isValidPos [] _ = False
isValidPos (column:_)  (0, y) = y >= 0 && length column > y
isValidPos (_:columns) (x, y) = isValidPos columns (x-1, y)

isFinished board = and $ concat $ mapBoard (\pt _ -> length (markBoard board pt) < 2) board

markBoard board (x, y) = markBoardMain (x, y) []
    where ball = board !! x !! y
          markBoardMain pt@(x, y) marks
              | isValidPos board pt && board !! x !! y == ball = 
                  foldr markBoardMain (pt:marks)
                        $ filter (flip notElem marks)
                              [(x+1, y), (x-1, y), (x, y+1), (x, y-1)]
              | otherwise = marks

pickBall :: Board -> [Pt] -> Board
pickBall board marks = filterBoard (\pt _ -> notElem pt marks) board

calcScore :: Board -> [Pt] -> Score
calcScore board marks = (length marks - 2) * (length marks - 2)


drawBoard vboard dc _ =
    do board <- varGet vboard
       set dc [brushKind := BrushSolid]
       mapM_ sequence_ $ mapBoard (\(x, y) c -> drawBall x y c) board
    where drawBall x y c = circle dc (pt (x*30+15) (285 - y*30)) 15 [brushColor := c]

colors = [red, blue, yellow, green, cyan, magenta, white]

mkRandomBoard :: Int -> Int -> Int -> IO Board
mkRandomBoard x y cols =
    do seed <- getStdGen
       rs <- return $ randomRs (0, cols-1) seed
       return $ take x $ mkTable y $ map (colors!!) rs
    where mkTable n list = column : mkTable n rest
              where (column, rest) = splitAt n list

pickEvent vboard vscore p scoreline pt =
    do board <- varGet vboard
       let (x, y) = (pointX pt `div` 30, 9 - pointY pt `div` 30)
           marks  = markBoard board (x, y)
       if length marks > 1
          then do varSet vboard $ pickBall board marks
                  varUpdate vscore (+ calcScore board marks)
                  score <- varGet vscore
                  set scoreline [text := "Score: " ++ show score]
                  repaint p
                  board <- varGet vboard
                  if isFinished board
                     then popupScore vscore
                     else return ()
          else return ()

popupScore vscore =
    do score <- varGet vscore
       f <- frame [text := "Your Score"]
       text1 <- staticText f [text := "Your Score is"]
       text2 <- staticText f [text := show score]
       okbutton <- button  f [text := "OK", on command := close f]
       
       set f [layout := column 0 $ map floatCenter [ widget text1
                                                   , widget text2
                                                   , widget okbutton]
             ]
       return ()

samegame =
    do vboard <- mkRandomBoard 15 10 3 >>= varCreate
       vscore <- varCreate 0
       f <- frame [text := "Wx SameGame"]
       field <- panel f [on paint := drawBoard vboard]
       scoreline <- staticText f [text := "Score: 0"]

       set field [on click         := pickEvent vboard vscore field scoreline
                 ,on (charKey 'q') := close f
                 ]
       set f [layout := column 0 [minsize (sz 450 300) $ widget field
                                 ,widget scoreline]
             ]
       return ()

main = start samegame
