package jpa.study.entity;

import static org.junit.Assert.assertEquals;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.persistence.TemporalType;

import org.junit.runner.RunWith;
import org.seasar.framework.unit.Seasar2;
import org.seasar.framework.unit.annotation.TxBehavior;
import org.seasar.framework.unit.annotation.TxBehaviorType;

/**
 * リスト5.1_1 Queryの基本操作を確かめるためのテストクラス
 */
@RunWith(Seasar2.class)
@TxBehavior(TxBehaviorType.COMMIT)
public class QueryTest {

	EntityManager em;

	/**
	 * リスト5.2_1 クエリ文字列を使った生成と実行
	 * 
	 * @throws Exception
	 */
	public void createQuery() throws Exception {
		String jpql = "select e from Emp e where e.sal < 1000 order by e.empName";
		Query query = em.createQuery(jpql);
		@SuppressWarnings("unchecked")
		List<Emp> list = query.getResultList();
		assertEquals(2, list.size());
		assertEquals("JAMES", list.get(0).getEmpName());
		assertEquals("SMITH", list.get(1).getEmpName());
	}

	/**
	 * リスト5.2_3 名前付きクエリを利用した生成と実行
	 * 
	 * @throws Exception
	 */
	public void createNamedQuery() throws Exception {
		Query query = em.createNamedQuery("QueryTest.createNamedQuery");
		@SuppressWarnings("unchecked")
		List<Emp> list = query.getResultList();
		assertEquals(2, list.size());
		assertEquals("JAMES", list.get(0).getEmpName());
		assertEquals("SMITH", list.get(1).getEmpName());
	}

	/**
	 * リスト5.3_1 パラメータの設定
	 * 
	 * @throws Exception
	 */
	public void parameter() throws Exception {
		String jpql = "select e from Emp e where e.sal < :sal order by e.empName";
		Query query = em.createQuery(jpql);
		query.setParameter("sal", new BigDecimal(1000));
		@SuppressWarnings("unchecked")
		List<Emp> list = query.getResultList();
		assertEquals(2, list.size());
		assertEquals("JAMES", list.get(0).getEmpName());
		assertEquals("SMITH", list.get(1).getEmpName());
	}

	/**
	 * リスト5.3_3 パラメータの設定（時間型）
	 * 
	 * @throws Exception
	 */
	public void parameter_temporalType() throws Exception {
		String jpql = "select e from Emp e where e.hiredate = :hiredate";
		Date hiredate = new SimpleDateFormat("yyyy-MM-dd").parse("1981-02-20");
		@SuppressWarnings("unchecked")
		List<Emp> list = em.createQuery(jpql).setParameter("hiredate",
				hiredate, TemporalType.DATE).getResultList();
		assertEquals(1, list.size());
		assertEquals("ALLEN", list.get(0).getEmpName());
	}

	/**
	 * リスト5.3_5 複数パラメータの設定
	 * 
	 * @throws Exception
	 */
	public void multiParameters() throws Exception {
		String jpql = "select e from Emp e where e.sal < :sal and e.empName like :empName";
		@SuppressWarnings("unchecked")
		List<Emp> list = em.createQuery(jpql).setParameter("sal",
				new BigDecimal(1000)).setParameter("empName", "S%")
				.getResultList();
		assertEquals(1, list.size());
		assertEquals("SMITH", list.get(0).getEmpName());
	}

	/**
	 * リスト5.3_7 複数パラメータ（順序パラメータ）
	 * 
	 * @throws Exception
	 */
	public void multiParameters_positional() throws Exception {
		String jpql = "select e from Emp e where e.sal < ?1 and e.empName like ?2";
		@SuppressWarnings("unchecked")
		List<Emp> list = em.createQuery(jpql).setParameter(1,
				new BigDecimal(1000)).setParameter(2, "S%").getResultList();
		assertEquals(1, list.size());
		assertEquals("SMITH", list.get(0).getEmpName());
	}

	/**
	 * リスト5.4_1 ページング
	 * 
	 * @throws Exception
	 */
	public void paging() throws Exception {
		String jpql = "select e from Emp e where e.sal > :sal order by e.empName";
		@SuppressWarnings("unchecked")
		List<Emp> list = em.createQuery(jpql).setParameter("sal",
				new BigDecimal(1000)).setFirstResult(3).setMaxResults(5)
				.getResultList();
		assertEquals(5, list.size());
		assertEquals("CLARK", list.get(0).getEmpName());
		assertEquals("FORD", list.get(1).getEmpName());
		assertEquals("JONES", list.get(2).getEmpName());
		assertEquals("KING", list.get(3).getEmpName());
		assertEquals("MARTIN", list.get(4).getEmpName());
	}

}
