package jpa.study.entity;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.List;

import javax.persistence.EntityManager;

import org.junit.runner.RunWith;
import org.seasar.framework.unit.Seasar2;
import org.seasar.framework.unit.annotation.TxBehavior;
import org.seasar.framework.unit.annotation.TxBehaviorType;

/**
 * リスト6.2_5 JpqlTestクラス
 */
@RunWith(Seasar2.class)
@TxBehavior(TxBehaviorType.COMMIT)
public class JpqlTest {

	EntityManager em;

	/**
	 * リスト6.4_2 エンティティを取得するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void select_entity() throws Exception {
		String jpql = "select e from Employee e";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(4, list.size());
	}

	/**
	 * リスト6.4_5 単一の状態フィールドを取得するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void select_oneField() throws Exception {
		String jpql = "select e.age from Employee e";
		@SuppressWarnings("unchecked")
		List<Integer> list = em.createQuery(jpql).getResultList();
		assertEquals(4, list.size());
		for (Integer age : list) {
			assertTrue(age instanceof Integer);
		}
	}

	/**
	 * リスト6.4_8 複数の状態フィールドを取得するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void select_multiFields() throws Exception {
		String jpql = "select e.name, e.age from Employee e";
		@SuppressWarnings("unchecked")
		List<Object[]> list = em.createQuery(jpql).getResultList();
		assertEquals(4, list.size());
		for (Object[] o : list) {
			assertEquals(2, o.length);
			assertTrue(o[0] instanceof String);
			assertTrue(o[1] instanceof Integer);
		}
	}

	/**
	 * リスト6.4_12 コンストラクタ式を利用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void select_constructor() throws Exception {
		String jpql = "select new jpa.study.entity.NameAgeDto(e.name, e.age) from Employee e";
		@SuppressWarnings("unchecked")
		List<NameAgeDto> list = em.createQuery(jpql).getResultList();
		assertEquals(4, list.size());
		for (NameAgeDto dto : list) {
			assertNotNull(dto.getName());
			assertNotNull(dto.getAge());
		}
	}

	/**
	 * リスト6.4_15 WHERE句を利用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void where() throws Exception {
		String jpql = "select e from Employee e where e.name = 'ALLEN' and e.age = 28";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("ALLEN", list.get(0).getName());
	}

	/**
	 * リスト6.4_22 Empty比較式を使ったJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void where_isEmpty() throws Exception {
		String jpql = "select d from Department d where d.employees is empty";
		@SuppressWarnings("unchecked")
		List<Department> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("SALES", list.get(0).getName());
		assertTrue(list.get(0).getEmployees().isEmpty());
	}

	/**
	 * リスト6.4_25 ORDER BY句を使用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void orderBy() throws Exception {
		String jpql = "select e from Employee e order by e.age desc, e.name";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(4, list.size());
		assertEquals("JONES", list.get(0).getName());
		assertEquals("WARD", list.get(1).getName());
		assertEquals("SMITH", list.get(2).getName());
		assertEquals("ALLEN", list.get(3).getName());
	}

	/**
	 * リスト6.4_28 GROUP BY句とHAVING句を利用したJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void aggregation() throws Exception {
		String jpql = "select max(e.age) from Employee e group by e.department having max(e.age) > 39";
		@SuppressWarnings("unchecked")
		List<Integer> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals(new Integer(40), list.get(0));
	}

	/**
	 * リスト6.4_31 副問い合わせを利用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void subquery() throws Exception {
		String jpql = " select e from Employee e where e.age = (select max(x.age) from Employee x)";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("JONES", list.get(0).getName());
	}

	/**
	 * リスト6.4_34 LOWER関数を利用したJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void function() throws Exception {
		String jpql = "select lower(e.name) from Employee e where e.name = 'ALLEN'";
		@SuppressWarnings("unchecked")
		List<String> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("allen", list.get(0));
	}

	/**
	 * リスト6.5_2 明示的な内部結合を利用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void innerJoin() throws Exception {
		String jpql = "select e from Employee e inner join e.department d where d.name = 'RESEARCH')";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("JONES", list.get(0).getName());
	}

	/**
	 * リスト6.5_7 外部結合を利用するJPQLから変換されるSQL
	 * 
	 * @throws Exception
	 */
	public void leftOuterJoin() throws Exception {
		String jpql = "select e from Employee e left outer join e.department d where d.name = 'RESEARCH')";
		@SuppressWarnings("unchecked")
		List<Employee> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		assertEquals("JONES", list.get(0).getName());
	}

	/**
	 * リスト6.5_12 DISTINCTを利用するJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void distinct() throws Exception {
		String jpql = "select distinct d from Department d inner join d.employees)";
		@SuppressWarnings("unchecked")
		List<Department> list = em.createQuery(jpql).getResultList();
		assertEquals(2, list.size());
	}

	/**
	 * リスト6.5_16 通常の結合を行うJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void nonFetch() throws Exception {
		String jpql = "select distinct d from Department d inner join d.employees where d.name = 'ACCOUNTING'";
		@SuppressWarnings("unchecked")
		List<Department> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		for (Employee e : list.get(0).getEmployees()) {
			System.out.println(e.getName());
		}
	}

	/**
	 * リスト6.5_19 フェッチ結合を行うJPQLの利用例
	 * 
	 * @throws Exception
	 */
	public void fetch() throws Exception {
		String jpql = "select distinct d from Department d inner join fetch d.employees where d.name = 'ACCOUNTING'";
		@SuppressWarnings("unchecked")
		List<Department> list = em.createQuery(jpql).getResultList();
		assertEquals(1, list.size());
		for (Employee e : list.get(0).getEmployees()) {
			System.out.println(e.getName());
		}
	}

}
